#!/usr/bin/perl

package Data::Preserve;
use v5.6;
use strict;
use warnings;
use Carp;
use Filter::Util::Call;

our $VERSION = '0.6';

sub import {
    my $self = shift;
    my $file = (caller)[1];
    filter_add( { file => $file });
} 

sub filter {
    my $self = shift;
    my $status;

    if (($status = filter_read()) > 0) { #we got something
	$_ = process_line( $_, $self->{'file'} );
    }

    return $status;
} 

my $re = qr/\bPRESERVE\s*\{\s*([\$\@%])([a-zA-Z_]\w*)\s*,\s*([a-zA-Z_]\w*)\s*\}/; 

sub process_line {
    my $line = shift;
    my $file = shift;
    return $line unless $line =~ /\bPRESERVE\b/;

    unless ($line =~ /$re/) {
	croak "usage: PRESERVE { \@variable, name } where variable can also be scalar, hash";
    } 

    $line =~ s/$re/my $1$2; BEGIN { my \$r = Data::Preserve::PRESERVE( '$file', '$3', \\$1$2 ); $1$2 = $1\$r if \$r }/g;
    return $line;
} 

our %vars;
sub PRESERVE {
    @_ == 3 or croak "usage: internal Preserve error: Data::Preserve::PRESERVE( filename, 'var_name', \$ref_to_var )";

    my $file = shift;
    my $name = shift;
    my $ref = shift;
    ref $ref or croak "internal Preserve error: PRESERVE is useless if given non-ref";
    
    my $return = $vars{$file}{$name};
    $vars{$file}{$name} = $ref;
    return $return;
}

1;

=head1 NAME

Data::Preserve - Perl module and filter to allow lexicals to preserve their value over reloads

=head1 SYNOPSIS

  use Data::Preserve;

  PRESERVE { $MONKEY, monkey }
  PRESERVE { @THINGS, stuff }
  PRESERVE {  %BOOK, words }

=head1 DESCRIPTION

Some programming tasks, such as monolithic servers that cannot be restarted without
serious interruption, require the use of dynamically reloadable modules if one wants to 
easily upgrade them.  In Perl, as long as the module consists only of subroutine definitions,
this works fine by simply using C<do>; the newly read subroutine is redefined over the old one
of the same name and all works well.  (There is a warning given telling of the redefined
sub, but this can be turned off with (under Perl v5.6) C<no warnings 'redefine'>.)
It is as if the following was in the original program:

    sub foo {  # old, will be redefined
	print "hey";
    } 

    sub foo { # will be used
	print "nay";
    }

However, there are further difficulties if the subroutines try to use lexical but static
variables.  For example, a typical static variable will be reloaded like so:

    {
	my $S;
	sub set_s { $S = shift } # old
	sub get_s { $S } # old
    } 

    {
	my $S;
	sub set_s { $S = shift } #used
	sub get_s { $S } #used
    } 

The two C<$S> variables are totally unrelated, so any changes to C<$S> between loads
will be lost!  One "solution" is to make $S a package variable instead of a lexical, but
this can lead to bugs as it no longer has lexical scope.

The solution: preserve the variables.  b<Data::Preserve> is one way to do this.  C<use Data::Preserve>
installs a source filter over the current file to process C<PRESERVE> blocks; it B<must> be
included in any file that uses C<PRESERVE> blocks.

A C<PRESERVE> block consists of the word C<PRESERVE>, an opening curly bracket, the name of
a previously-undeclared scalar, array, or hash, a comma, an I<identifier>, and a closing curly
bracket.  The C<PRESERVE> block is effectively a declaration of the variable, so do not
declare it elsewhere.  The identifier gives the B<Data::Preserve> module its only hint as to how
the preserved variables are connected; it need not be the same as the name of the variable,
but unlike the variable name it must be unique within the current B<file>.

Effectively, a C<PRESERVE> block declares its variable and calls a function in the B<Data::Preserve>
module that stores a reference to the new variable indexed by filename and the given I<identifier>,
and then if there was a previous C<PRESERVE> of the same variable (that is, I<identifier> in the
current file), it returns the previous variable's reference and dereferences it into the new variable.
But it is not important to worry about the inner workings of B<Data::Preserve>.

If you need to preserve something other than a scalar, array, or hash, you can preserve
a reference to it, which is a scalar and thus can be preserved normally.

=head1 BUGS

A C<PRESERVE> block must fit all onto one line.

You cannot usefully take the reference of a variable that is C<PRESERVE>d and keep the
reference over the course of a reload; an attempt to do so will lead to confusion,
making the reference refer to a different variable than naming the variable will.
However, C<PRESERVE>d variables can B<contain> references, and in fact any 
such reference will be preserved unchanged.

=head1 PREREQUISITES

B<Data::Preserve> requires the B<Filter> package from CPAN and Perl v5.6.

=head1 AUTHOR

David Glasser, me@davidglasser.net

=cut
