#!/usr/local/bin/perl -w
# vim: ts=4
use strict;
use constant RCS_ID => '$Id: webly.pl,v 2.12 2020/02/21 20:15:43 inky Exp $';

# $Log: webly.pl,v $
# Revision 2.12  2020/02/21 20:15:43  inky
# Fixing undefined error
#
# Revision 2.11  2007/02/02 00:21:46  inky
# Encode html sent to user
#
# Revision 2.10  2006/02/20 00:34:33  inky
# Add content header (to help IE) - inky
#
# Revision 2.9  2006/02/18 03:23:40  ifmud
# added xml header in an attempt to make IE6 happy
#
# Revision 2.8  2006/02/16 04:14:51  ifmud
# modifications to support ajax UI
#
# Revision 2.7  2005/04/04 16:51:55  jota
# Adjusted the auto-generated URL's to match the format the MUD wants
#
# Revision 2.6  2002/12/05 23:32:03  inky
# Doesn't seem to be correctly handling the initial redirect (or, rather, it's
# handling it but redirecting it to the "real" host, when I'd like it to stay
# as ifmud.port4000.com). So make the redirect explicit.
#
# Revision 2.5  2001/01/12 02:21:20  david
# Add local port control for mud sockets.
#
# Revision 2.4  2001/01/12 01:56:22  david
# Close MUD sockets on their disconnectoin.
#
# Revision 2.3  2000/10/24 03:18:45  david
# select on clientconns
#
# Revision 2.2  2000/10/14 22:14:41  david
# Removed list of subs that hasn't been updated and is
# easily removed by grep; added RCS log.
#

BEGIN {
	@ARGV == 3 or die "usage: webly.pl datadir web_port mud_port";
	my $data = shift;
	my $wp   = shift;
	my $mp   = shift;
	eval qq(
		use constant DATA_DIR => '$data';
		use constant  MY_PORT => $wp;
		use constant MUD_PORT => $mp;
		1;
	) or die $@;
}

BEGIN {
	local *F;
	my $fname = DATA_DIR . "webly.pid";
	open F, ">$fname" or die "can't open $fname for writing: $!\n";
	print F $$;
	close F;
} 

use HTTP::Daemon;
use HTTP::Status;
use IO::Select;
use URI::Escape ();

$SIG{'PIPE'} = 'IGNORE';

use constant MY_HOST		=> 'ifmud.port4000.com';
use constant MUD_HOST		=> 'localhost'; # It won't really work on non-local hosts anyway
use constant FRONT_PAGE		=> DATA_DIR . 'webly_front.html';
use constant BUI_PAGE       => DATA_DIR . 'mudbui.html';
use constant DOJO_DIR       => '/opt/ifmud/web/dojo/'; # need trailing slash
use constant ROW_DEFAULT	=> 100;
use constant WRAP_LENGTH	=> 100;
use constant IDLE_TIME		=> ( 60 * 20 ); # Seconds until disconnection.  So, 20 minutes.
# range of local ports to use for the mud connections.
use constant LPORT_FIRST    => 10400;
use constant LPORT_LAST     => 10499;

use constant NO_PLAYER		=> <<END_TEXT;
Sorry, I can't find your session.
You can <a href="/connect" target="_top">make a new connection</a>.
END_TEXT
									#'

use constant SOCKET_DAEMON => 1;
use constant SOCKET_MUD    => 2;
use constant SOCKET_WEB    => 3;

MAIN: {
	STDOUT->autoflush(1);
	my $daemon = HTTP::Daemon->new( LocalPort => MY_PORT, LocalAddr => MY_HOST, Reuse => 1 ) or die "can't make daemon: $!\n";
	my $select = IO::Select->new( [ $daemon, SOCKET_DAEMON ] );
	sub add_to_selection { #used for mud sockets
		my $socket = shift;
		$select->add([ $socket, SOCKET_MUD ] );
	}
	sub remove_from_selection {
		my $socket = shift;
		$select->remove($socket);
	} 
	print "daemon started at ", $daemon->url, "\n";

	while (1) {
		my $next_up = IdleQueue::front();
		my $timeout = $next_up ? $next_up->timeout : undef;
		my @ready = $select->can_read($timeout);
		for my $sock (@ready) {
			if ($sock->[1] == SOCKET_DAEMON) {
				if  (my $client = $daemon->accept) {
					$select->add( [ $client, SOCKET_WEB ]);
				}
			} elsif ($sock->[1] == SOCKET_MUD) {
				handle_mud_read(Pool::sock2player($sock->[0]));
			} else { # SOCKET_WEB
				my $do_close;
				if (my $request = $sock->[0]->get_request) {
					$do_close = handle_request($sock->[0], $request);
				}
				$select->remove($sock->[0]);
				if ($do_close) {
					$sock->[0]->close;
				} 
			} 
		}

		# After from reading from all that can be read,
		# check for over-idleness
		my $tardiest = IdleQueue::front();
		if ($tardiest and $tardiest->too_idle) {
			$tardiest->force_quit;
			IdleQueue::remove_front();
		} 
	} 
}

{
	package Pool;
	my %POOL;
	my %SOCK2PLAYER;

	sub add_player {
		my $socket = shift;
        my $isXML = shift;

		my $key;
		do { $key = join '', map { ( 'a' .. 'z' )[ rand 26 ] } 1 .. 15 } until not current_key($key);
		$POOL{ $key } = Player->new( $socket, $key, $isXML );
		$SOCK2PLAYER{ $socket } = $POOL{ $key };
		return $POOL{ $key };
	}

	sub remove_player {
		my $key = shift;
		delete $SOCK2PLAYER{ $POOL{ $key }->get_socket } if $POOL{ $key };
		delete $POOL{ $key };
	} 

	sub get_player {
		my $key = shift;
		return $POOL{ $key };
	}

	sub current_key {
		my $key = shift;
		return defined $POOL{ $key };
	}

	sub sock2player {
		my $sock = shift;
		return $SOCK2PLAYER{ $sock };
	} 
}

{
	package IdleQueue;
	my @QUEUE;

	# Operations on the queue:
	# - Insert-at-back (&queue)
	#     (for new players and for after commands)
	# - Return-front (&front)
	#     (to check to see if the front is old enough)
	# - Remove-front (&remove_front)
	#     (if it was; this doesn't return it)
	# - Remove-player-from-center (&remove_player)
	#     (this searches (linearly) for the player and splices it out,
	#      for after commands or on quits; no return)

	sub queue {
		my $item = shift;

		push @QUEUE, $item;
		return;
	}

	sub front {
		return @QUEUE ? $QUEUE[0] : undef;
	}

	sub remove_front {
		shift @QUEUE if @QUEUE;
		return;
	}

	sub remove_player {
		my $item = shift;
		my $found = undef;
		SEARCH: for (my $i = 0; $i < @QUEUE; $i++) {
			next SEARCH unless $QUEUE[$i] == $item;
			$found = $i;
			last SEARCH;
		} 
		if (defined $found) {
			splice @QUEUE, $found, 1;
		} 
		return;
	} 
} 

{
	package Player;
	
	sub new {
		my $class = shift;
		my $socket = shift;
		my $key = shift;
        my $isXML = shift;

		bless { Socket => $socket, Buffer => [], Key => $key, IsXML => $isXML, Trailer => "" }, $class;
	}

	sub get_socket {
		my $self = shift;
		return $self->{'Socket'};
	} 

	sub get_key {
		my $self = shift;
		return $self->{'Key'};
	}

    sub is_xml {
        my $self = shift;
        return $self->{'IsXML'};
    }

	sub add_buffer {
		my $self = shift;
		my $line = shift;
		$line =~ s/[\r\n]+$//;

		my @lines = split /\n/, join "\n", ::wrap_line($line);
		@lines = '' unless @lines;
		
		for my $l (@lines) {
                  $l =~ s/&/&amp;/g if $self->is_xml;
			for my $re (@{ $self->{'highlights'} }) {
                if ($self->is_xml) {
                    $l =~ s#$re#\&lt;b\&gt;$1\&lt;/b\&gt;#g;
                } else {
                    $l =~ s#$re#<b>$1</b>#g;
                }
			} 
			$l =~ s/  \x01
		           ( [^\x02\x03] + )
				     \x03
				   ( [^\x02\x03] + )
				     \x02
				  /$self->command_link($1, $2)/xeg;
		}

		push @{ $self->{'Buffer'} }, @lines;
	}

	sub command_link {
		my $self = shift;
		my $text = shift;
		my $link = shift;

		my $key = $self->get_key;

		$link =~ s#<b>##g; $link =~ s#</b>##g; # no highlight!

        if ($self->is_xml) {
            $link =~ s#'#\\'#g;
            return qq#&lt;a href="javascript:void(0);" onclick="postCommand('$link');"&gt;$text&lt;/a&gt;#;
        } else {
            $link = URI::Escape::uri_escape($link);
            return qq(<a href="/upper/$key?command=$link&update=Go#newest">$text</a>);
        }
	} 

	sub get_buffer {
		my $self = shift;
		return @{ $self->{'Buffer'} };
	}

	sub insert_pound {
		my $self = shift;
        unless ($self->is_xml) {
            my $code = '<a name=newest><b>#</b></a>';
            @{ $self->{'Buffer'} } = grep $_ ne $code, @{ $self->{'Buffer'} };
            push @{ $self->{'Buffer'} }, $code;
        }
	} 

    sub clear_buffer {
        my $self = shift;
        @{ $self->{'Buffer'} } = ();
    }

	sub limit_buffer {
		my $self = shift;
		my $limit = shift;
		my $amount_to_destroy = @{ $self->{'Buffer'} } - $limit;
		return unless $amount_to_destroy > 0;
		splice @{ $self->{'Buffer'} }, 0, $amount_to_destroy;
	}
	
	sub waiting {
		my $self = shift;
		return $self->{'is_waiting'} ? 1 : 0;
	} 

	sub start_wait { # it's ok to do this while still waiting
		my $self   = shift;
		my $client = shift;
		my $key    = shift;

		$self->{'is_waiting'}  = 1;
		$self->{'wait_params'} = [ $client, $key ];
	} 

	sub end_wait {
		my $self = shift;
		
		$self->{'is_waiting'} = 0;
        if ($self->is_xml) {
            ::handle_upper_xml(@{ $self->{'wait_params'} });
        } else {
            ::handle_upper(@{ $self->{'wait_params'} });
        }
		undef $self->{'wait_params'};
	}

	sub set_disconnected {
		my $self = shift;
		$self->{'disconnected'} = 1;
	}

	sub disconnected {
		my $self = shift;
		return $self->{'disconnected'} ? 1 : 0;
	} 

	sub not_idle {
		my $self = shift;
		$self->{'last_time'} = time;
		IdleQueue::remove_player($self);
		IdleQueue::queue($self);
	} 

	sub too_idle {
		my $self = shift;
		my $delta = (time) - $self->{'last_time'};
		if ($delta > ::IDLE_TIME) {
			return 1;
		} else {
			return 0;
		} 
	} 

	sub timeout {
		my $self = shift;
		my $last = $self->{'last_time'};
		my $delta = (time) - $last;
		my $timeleft = ::IDLE_TIME - $delta;
		$timeleft = 0 if $timeleft < 0;
		return $timeleft;
	} 

	sub force_quit {
		my $self = shift;
		
		$self->run_command('@quit');
	} 

	sub run_command {
		my $self = shift;
		my $command = shift;
		my $socket = $self->get_socket;
		$socket->print($command . "\n");
	} 

	sub add_highlight {
		my $self = shift;
		my $re   = shift;
		push @{ $self->{'highlights'} }, $re;
	} 

    sub set_trailer {
        my $self = shift;
        my $t    = shift;
        $self->{'Trailer'} = $t;
    }

    sub get_trailer {
        my $self = shift;
        my $result = $self->{'Trailer'};
        $self->{'Trailer'} = "";
        return $result;
    }
} 

sub respond {
	my $client = shift;
	my $text   = shift;

	my $response = HTTP::Response->new(RC_OK);
	$response->content($text);
	$client->send_response($response);
} 

sub parse_request_params {
	my $request = shift;
	my $parser = $request->url->clone;
	$parser->equery($request->content) unless defined $parser->equery and length $parser->equery;
	return $parser->query_form; # returns a hash of params
} 

sub handle_request {
	my $client  = shift;
	my $request = shift;
	if ($request->url->path eq '/' or $request->url->path eq '/index.html') {
		$client->send_file_response( FRONT_PAGE );
	} elsif ($request->url->path eq '/connect') {
		my %params = parse_request_params($request);
		handle_new_connection($client, 0, $request, $params{'old'});
    } elsif ($request->url->path eq '/bui') {
        $client->send_file_response( BUI_PAGE );
    } elsif ($request->url->path eq '/generateKey') {
        my %params = parse_request_params($request);
        handle_new_connection($client, 1, $request, $params{'old'});
    } elsif ($request->url->path =~ m#^/dojo/(.*)$#) {
        $client->send_file_response(DOJO_DIR . $1);
	} elsif ($request->url->path =~ m#^/upper/([a-z]{15})$#) {
		my $key = $1;
		my %params = parse_request_params($request);
		my $player = Pool::get_player($key);
        if (!$player) {
            # attempting to use invalid key
            $client->send_redirect( "http://" . MY_HOST . ":" . MY_PORT . "/", 302);
            return 1;
        }

		if ($player and not $player->disconnected) {
			$player->not_idle;
		} 

		if (defined $params{'command'} and length $params{'command'}
		     and $player and not $player->disconnected) {
			$player->run_command($params{'command'});
			$player->start_wait($client, $key);
			return 0; # This keeps the connection from being closed.
		} elsif ($player and not $player->disconnected) {
            if ($player->is_xml) {
                handle_upper_xml($client, $key);
            } else {
                handle_upper($client, $key);
            }
		} 
	} elsif ($request->url->path =~ m#^/lower/([a-z]{15})$#) {
		my $key = $1;
		handle_lower($client, $key);
	} elsif ($request->url->path =~ m#^/main/([a-z]{15})$#) {
		my $key = $1;
		send_mainpage($client, $key);
	} else {
		$client->send_error(RC_NOT_IMPLEMENTED);
	}

	return 1; # important!
} 

sub open_mud_socket {
	for (my $port = LPORT_FIRST; $port <= LPORT_LAST; $port++) {
		my $socket = IO::Socket::INET->new(
			PeerAddr => MUD_HOST,
			PeerPort => MUD_PORT,
			LocalPort => $port);
		return $socket if $socket;
	} 
	return 0; #maybe the mud is down, or there are no more local ports
} 

sub handle_new_connection {
	my $client  = shift;
    my $isXML   = shift;
	my $request = shift;
	my $old_key = shift;  # If this is a post-disconnect reconnect, we can clean up.
	# First, let's clean up an old Player.
	my $old_player;
	if (defined $old_key and $old_player = Pool::get_player($old_key)
	    and $old_player->disconnected) {
		Pool::remove_player($old_player);
	} 

	my $mud_socket = open_mud_socket();

	unless ($mud_socket) {
		$client->send_error(RC_INTERNAL_SERVER_ERROR, <<END_DESC);
Can't connect to ifMUD: $!. (Perhaps the MUD isn't running.)
END_DESC
		return;
	}

	$mud_socket->blocking(0);

	my $player = Pool::add_player( $mud_socket, $isXML );
	add_to_selection($mud_socket);
	$player->not_idle;

	my $addr_for_from = gethostbyaddr( $client->peeraddr, AF_INET );
	$addr_for_from    = $client->peerhost if !$addr_for_from;
	$addr_for_from =~ s/\s/_/g;
	$player->run_command("webly $addr_for_from");
	
	my $key = $player->get_key;
    if ($player->is_xml) {
        my $response = HTTP::Response->new(RC_OK);
        $response->header('Content-Type' => 'text/xml');
        $response->content('<?xml version="1.0" ?><data><key>' . $key . '</key></data>');
        $client->send_response($response);
    } else {
        $client->send_redirect( "http://" . MY_HOST . ":" . MY_PORT . "/main/$key" );
    }
}

sub handle_upper {
	my $client = shift;
	my $key    = shift;

	my $player = Pool::get_player($key);

	unless ($player) {
		$client->send_error(RC_NOT_FOUND, NO_PLAYER);
		return;	
	}

	my $page = <<END_HTML;
<html>
<head>
<meta http-equiv="refresh" content="15;URL=/upper/$key#newest">
</head>
<body>
<pre>
END_HTML
                                         #"
	for my $line ($player->get_buffer) {
		$page .= $line . "\n";
	}
	$page .= <<END_HTML;
</pre>
</body>
</html>
END_HTML
	my $response = HTTP::Response->new(RC_OK);
	$response->content($page);
#	$response->push_header('Refresh', "6");
	$client->send_response($response);
}

sub handle_upper_xml {
    my $client = shift;
    my $key    = shift;
    my $player = Pool::get_player($key);

    unless ($player) {
        $client->send_error(RC_NOT_FOUND, NO_PLAYER);
    }

    my $response = HTTP::Response->new(RC_OK);
    $response->header('Content-Type' => 'text/xml');
    my $linedata = '<?xml version="1.0" ?><data>';
    for my $line ($player->get_buffer) {
        $linedata .= "<line>$line</line>";
    }
    $player->clear_buffer;
    $linedata .= "</data>";
    $response->content($linedata);
    $client->send_response($response);
}
sub handle_lower { 
	my $client = shift;
	my $key    = shift;

	my $player = Pool::get_player($key);

	unless ($player) {
		$client->send_error(RC_NOT_FOUND, NO_PLAYER);
		return;
	}

	my $page = <<END_HTML;
<html>
<head></head>
<body>
<table><tr><!-- keep all on one line, using table -->
<form name="commands" action="/upper/$key#newest" target="upper" method="GET" onSubmit="queueClear()">
	<td><input type="text" size=40 name="command"></td> <td><input type="submit" value="Go" name="update"></td>
</form>
<form name="refresh"  action="/upper/$key#newest" target="upper" method="GET">
	<td><input type="hidden" name="command" value="">
	    <input type="submit" value="Refresh upper frame" name="refresh"></td>
</form></tr></table>
<script>
function queueClear () {
	setTimeout('clearCommand()', 500)
	return 1
}
function clearCommand () {
	document.commands.command.value = ""
}
</script>
</body>
</html>
END_HTML
                              #"
	respond($client, $page);
}
sub send_mainpage {
	my $client = shift;
	my $key = shift;

	respond($client, <<END_HTML);
<html>
<head><title>ifMUD Webly</title></head>
<frameset rows="*,60" onLoad="frames[1].document.commands.command.focus();">
<frame name="upper" marginheight="1" src="/upper/$key#newest">
<frame name="lower" marginheight="1" src="/lower/$key">
</frameset>
</html>
END_HTML
}

sub handle_mud_read {
	my $player = shift;
	my $socket = $player->get_socket;

    my $data;
    my $datalen = $socket->read($data, 65536);

	unless ($datalen) {
		remove_from_selection($socket);
		my $key = $player->get_key;
		$player->add_buffer( 'You have been disconnected.  ' .
			qq(<a href="/connect?old=$key" target="_top">Click this to reconnect.<a>));
		$player->set_disconnected;
		IdleQueue::remove_player($player);
		$socket->close;
		return;
	} 

    $data = $player->get_trailer . $data;
	my @lines = split /\n/, $data;

    unless ($lines[-1] =~ /[\r]$/) {
        $player->set_trailer(pop @lines);
    }
	
	my $completed_command = 0;
	
	$player->insert_pound; # relocate the '#'
	
	LINES: for my $line (@lines) {
		$line =~ s/[\r\n]+$//;

		if ($line eq '.') {
			$completed_command = 1;
			$line = '';
		} elsif ($line =~ /^!/) {
			$line = substr $line, 1;
			my $re;
			eval { $re = qr/($line)/; };
			next LINES if $@; # don't even try to complain
			$player->add_highlight($re);
			next LINES;
		} 

		$line =~ s/^[-+]//;
	
		$player->add_buffer(html_encode($line));
	}
	$player->limit_buffer(ROW_DEFAULT);
	
	if ($completed_command and $player->waiting) {
		$player->end_wait;
	} 
}

sub html_encode {
	my $text = shift;
	
	$text =~ s/&/&amp;/g;
	$text =~ s/</&lt;/g;
	$text =~ s/>/&gt;/g;
	
	return $text;
}

sub wrap_line {
	my $line = shift;
	my @lines;
	WRAP: while (plainlength($line) > WRAP_LENGTH) {
		my $at = plainrindex($line, " ", WRAP_LENGTH);
		if ($at == -1) {
			last WRAP;
		}
		push @lines, substr( $line, 0, $at );
		$line = substr($line, $at + 1);
	}

	push @lines, $line if length $line;

	return @lines;
}

sub plainlength {
	my $arg = shift;
	$arg =~ s/\x01([^\,\x02]+)\x03([^\,\x02]+)\x02/$1/g;
	return length($arg);
}

sub plainrindex
{
        my($in, $sub, $last) = @_;
        my($foo) = substr($in, 0, $last);
        my($end, $break, $lat, $gat, $plen);
        $end = 0;
		$plen = 0;
        # Count 'last' non-escaped characters.
        while (1) {
                $lat = index($in, "\x01", $end);
                if ($lat == -1) {
                        $last = $end + ($last - $plen);
                        last;
                }
                if (($plen + ($lat - $end)) >= $last) {
                        $last = $end + ($last - $plen);
                        last;
                }
                $plen += ($lat - $end);
                $gat = index($in, "\x02", $lat + 1);
                if ($gat == -1) {
                        # Uh-oh, play dumb
                        return -1;
                }
                my($cat);
                $cat = index($in, "\x03", $lat + 1);
                if ($cat == -1) {
                        # Bad craziness.
                        return -1;
                }
                $plen += ($cat - $lat - 1);
                if ($plen >= $last) {
                        $last = $lat;
                        last;
                }
                $end = $gat + 1;
        }
        # Okay, now we know where the real limiting point is...
        my $at = rindex($in, $sub, $last);
        # Hackery to ensure we never break up an embedded link
        while ($at != -1) {
                $gat = index($in, "\x02", $at);
                $lat = index($in, "\x01", $at);
                if ($gat == -1) {
                        last;
                }
                if (($lat == -1) || ($gat < $lat)) {
                        if ($at != 0) {
                                $at = rindex(
                                        $in, $sub, $at - 1);
                        } else {
                                $at = -1;
                        }
                } else {
                        last;
                }
        }
        return $at;
}

